import { useState, useMemo } from "react";
import { DailySale, StockItem } from "@/data/stockData";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { ChevronDown, ChevronRight, Plus, Trash2, CalendarIcon, Filter, Search, FileText } from "lucide-react";
import { ToggleGroup, ToggleGroupItem } from "@/components/ui/toggle-group";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Calendar } from "@/components/ui/calendar";
import { cn } from "@/lib/utils";
import { format, parse, isWithinInterval, startOfDay, endOfDay, startOfWeek, endOfWeek, startOfMonth, endOfMonth } from "date-fns";
import AddDailySaleDialog from "./AddDailySaleDialog";
import InvoiceDialog from "./InvoiceDialog";
import { useToast } from "@/hooks/use-toast";

const parseDateStr = (dateStr: string): Date => {
  return parse(dateStr, "dd/MM/yyyy", new Date());
};

type DatePreset = "all" | "today" | "week" | "month" | "custom";

interface DailySalesTableProps {
  sales: DailySale[];
  stockItems: StockItem[];
  onAddSale: (sale: DailySale, stockOutItems: { itemName: string; qty: number }[]) => void;
  onDeleteSale: (date: string) => void;
}

const DailySalesTable = ({
  sales,
  stockItems,
  onAddSale,
  onDeleteSale,
}: DailySalesTableProps) => {
  const { toast } = useToast();
  const [expandedDate, setExpandedDate] = useState<string | null>(null);
  const [addOpen, setAddOpen] = useState(false);
  const [invoiceDialogOpen, setInvoiceDialogOpen] = useState(false);
  const [selectedSale, setSelectedSale] = useState<DailySale | null>(null);
  const [datePreset, setDatePreset] = useState<DatePreset>("all");
  const [customFrom, setCustomFrom] = useState<Date | undefined>();
  const [customTo, setCustomTo] = useState<Date | undefined>();
  const [productFilter, setProductFilter] = useState("");

  const allProductNames = useMemo(() => {
    const names = new Set<string>();
    sales.forEach((s) => s.items.forEach((i) => names.add(i.itemName)));
    return Array.from(names).sort();
  }, [sales]);

  const filteredSales = useMemo(() => {
    let result = sales;

    // Date filtering
    const today = new Date();
    let fromDate: Date | undefined;
    let toDate: Date | undefined;

    if (datePreset === "today") {
      fromDate = startOfDay(today);
      toDate = endOfDay(today);
    } else if (datePreset === "week") {
      fromDate = startOfWeek(today, { weekStartsOn: 6 });
      toDate = endOfWeek(today, { weekStartsOn: 6 });
    } else if (datePreset === "month") {
      fromDate = startOfMonth(today);
      toDate = endOfMonth(today);
    } else if (datePreset === "custom") {
      fromDate = customFrom ? startOfDay(customFrom) : undefined;
      toDate = customTo ? endOfDay(customTo) : undefined;
    }

    if (fromDate || toDate) {
      result = result.filter((s) => {
        const d = parseDateStr(s.date);
        if (fromDate && toDate) return isWithinInterval(d, { start: fromDate, end: toDate });
        if (fromDate) return d >= fromDate;
        if (toDate) return d <= toDate;
        return true;
      });
    }

    // Product filtering
    if (productFilter.trim()) {
      const q = productFilter.toLowerCase();
      result = result.filter((s) =>
        s.items.some((i) => i.itemName.toLowerCase().includes(q))
      );
    }

    return result;
  }, [sales, datePreset, customFrom, customTo, productFilter]);

  const handleAddSale = (
    sale: DailySale,
    stockOutItems: { itemName: string; qty: number }[]
  ) => {
    onAddSale(sale, stockOutItems);
    toast({
      title: "Sale Recorded",
      description: `${sale.items.length} items sold on ${sale.date}. Stock has been updated.`,
    });
  };

  const handleDelete = (date: string) => {
    onDeleteSale(date);
    toast({
      title: "Sale Deleted",
      description: `Sale record for ${date} removed.`,
      variant: "destructive",
    });
  };

  return (
    <>
      <div
        className="glass-card rounded-xl overflow-hidden animate-fade-in"
        style={{ animationDelay: "500ms" }}
      >
        <div className="p-5 border-b border-border space-y-3">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3">
            <div>
              <h3 className="text-lg font-bold font-display text-foreground">
                Daily Sales Report
              </h3>
              <p className="text-sm text-muted-foreground mt-1">
                {filteredSales.length} of {sales.length} records
              </p>
            </div>
            <Button onClick={() => setAddOpen(true)} size="sm" className="gap-1.5">
              <Plus className="h-4 w-4" />
              <span className="hidden sm:inline">Record Sale</span>
            </Button>
          </div>

          {/* Filters */}
          <div className="flex flex-col sm:flex-row gap-3 items-start sm:items-center">
            <ToggleGroup
              type="single"
              value={datePreset}
              onValueChange={(v) => v && setDatePreset(v as DatePreset)}
              size="sm"
              className="flex-wrap"
            >
              <ToggleGroupItem value="all" className="text-xs px-3">All</ToggleGroupItem>
              <ToggleGroupItem value="today" className="text-xs px-3">Today</ToggleGroupItem>
              <ToggleGroupItem value="week" className="text-xs px-3">This Week</ToggleGroupItem>
              <ToggleGroupItem value="month" className="text-xs px-3">This Month</ToggleGroupItem>
              <ToggleGroupItem value="custom" className="text-xs px-3">Custom</ToggleGroupItem>
            </ToggleGroup>

            <div className="relative">
              <Search className="absolute left-2.5 top-1/2 -translate-y-1/2 h-3.5 w-3.5 text-muted-foreground" />
              <Input
                placeholder="Filter by product..."
                value={productFilter}
                onChange={(e) => setProductFilter(e.target.value)}
                className="pl-8 h-8 text-xs w-48"
              />
            </div>
          </div>

          {/* Custom date range */}
          {datePreset === "custom" && (
            <div className="flex flex-wrap gap-2 items-center">
              <Popover>
                <PopoverTrigger asChild>
                  <Button variant="outline" size="sm" className={cn("text-xs gap-1.5", !customFrom && "text-muted-foreground")}>
                    <CalendarIcon className="h-3.5 w-3.5" />
                    {customFrom ? format(customFrom, "dd/MM/yyyy") : "From"}
                  </Button>
                </PopoverTrigger>
                <PopoverContent className="w-auto p-0" align="start">
                  <Calendar mode="single" selected={customFrom} onSelect={setCustomFrom} initialFocus className="p-3 pointer-events-auto" />
                </PopoverContent>
              </Popover>
              <span className="text-xs text-muted-foreground">to</span>
              <Popover>
                <PopoverTrigger asChild>
                  <Button variant="outline" size="sm" className={cn("text-xs gap-1.5", !customTo && "text-muted-foreground")}>
                    <CalendarIcon className="h-3.5 w-3.5" />
                    {customTo ? format(customTo, "dd/MM/yyyy") : "To"}
                  </Button>
                </PopoverTrigger>
                <PopoverContent className="w-auto p-0" align="start">
                  <Calendar mode="single" selected={customTo} onSelect={setCustomTo} initialFocus className="p-3 pointer-events-auto" />
                </PopoverContent>
              </Popover>
            </div>
          )}
        </div>
        <div className="overflow-x-auto">
          <Table>
            <TableHeader>
              <TableRow className="bg-muted/50">
                <TableHead className="w-8"></TableHead>
                <TableHead className="font-semibold text-foreground">Date</TableHead>
                <TableHead className="text-right font-semibold text-foreground">
                  Sales (৳)
                </TableHead>
                <TableHead className="text-right font-semibold text-foreground">
                  Cost (৳)
                </TableHead>
                <TableHead className="text-right font-semibold text-foreground">
                  Profit (৳)
                </TableHead>
                <TableHead className="text-right font-semibold text-foreground">
                  Paid (৳)
                </TableHead>
                <TableHead className="text-right font-semibold text-foreground">
                  Due (৳)
                </TableHead>
                <TableHead className="text-right font-semibold text-foreground">
                  Net Profit (৳)
                </TableHead>
                <TableHead className="w-10"></TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredSales.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={9} className="text-center py-8 text-muted-foreground">
                    {sales.length === 0 ? 'No sales recorded yet. Click "Record Sale" to add one.' : "No sales match the current filters."}
                  </TableCell>
                </TableRow>
              ) : (
                filteredSales.map((day) => (
                  <>
                    <TableRow
                      key={day.date}
                      className="hover:bg-muted/30 cursor-pointer transition-colors"
                      onClick={() =>
                        setExpandedDate(expandedDate === day.date ? null : day.date)
                      }
                    >
                      <TableCell className="text-muted-foreground">
                        {expandedDate === day.date ? (
                          <ChevronDown className="h-4 w-4" />
                        ) : (
                          <ChevronRight className="h-4 w-4" />
                        )}
                      </TableCell>
                      <TableCell className="font-medium text-foreground">
                        {day.date}
                      </TableCell>
                      <TableCell className="text-right font-semibold text-primary">
                        ৳{day.totalTP.toLocaleString()}
                      </TableCell>
                      <TableCell className="text-right text-muted-foreground">
                        ৳{day.totalDP.toLocaleString()}
                      </TableCell>
                      <TableCell className="text-right font-semibold text-success">
                        ৳{day.totalProfit.toLocaleString()}
                      </TableCell>
                      <TableCell className="text-right text-muted-foreground">
                        ৳{day.paidAmount.toLocaleString()}
                      </TableCell>
                      <TableCell className="text-right text-destructive">
                        {day.dueAmount > 0 ? `৳${day.dueAmount.toLocaleString()}` : "—"}
                      </TableCell>
                      <TableCell className="text-right font-bold text-foreground">
                        ৳{day.netProfit.toLocaleString()}
                      </TableCell>
                      <TableCell>
                        <div className="flex items-center gap-1">
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-7 w-7 text-primary hover:text-primary hover:bg-primary/10"
                            title="View Invoice"
                            onClick={(e) => {
                              e.stopPropagation();
                              setSelectedSale(day);
                              setInvoiceDialogOpen(true);
                            }}
                          >
                            <FileText className="h-3.5 w-3.5" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-7 w-7 text-destructive hover:text-destructive hover:bg-destructive/10"
                            title="Delete"
                            onClick={(e) => {
                              e.stopPropagation();
                              handleDelete(day.date);
                            }}
                          >
                            <Trash2 className="h-3.5 w-3.5" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                    {expandedDate === day.date && (
                      <TableRow key={`${day.date}-details`}>
                        <TableCell colSpan={9} className="p-0 bg-muted/20">
                          <div className="p-4">
                            <Table>
                              <TableHeader>
                                <TableRow className="border-b border-border/50">
                                  <TableHead className="text-xs text-muted-foreground font-semibold">
                                    Item
                                  </TableHead>
                                  <TableHead className="text-right text-xs text-muted-foreground font-semibold">
                                    Qty
                                  </TableHead>
                                  <TableHead className="text-right text-xs text-muted-foreground font-semibold">
                                    Cost
                                  </TableHead>
                                  <TableHead className="text-right text-xs text-muted-foreground font-semibold">
                                    Price
                                  </TableHead>
                                  <TableHead className="text-right text-xs text-muted-foreground font-semibold">
                                    Total
                                  </TableHead>
                                  <TableHead className="text-right text-xs text-muted-foreground font-semibold">
                                    Profit
                                  </TableHead>
                                </TableRow>
                              </TableHeader>
                              <TableBody>
                                {day.items.map((item, i) => (
                                  <TableRow key={i} className="border-b border-border/30">
                                    <TableCell className="text-sm text-foreground">
                                      {item.itemName}
                                    </TableCell>
                                    <TableCell className="text-right text-sm text-muted-foreground">
                                      {item.qty}
                                    </TableCell>
                                    <TableCell className="text-right text-sm text-muted-foreground">
                                      ৳{item.dp}
                                    </TableCell>
                                    <TableCell className="text-right text-sm text-muted-foreground">
                                      ৳{item.tp}
                                    </TableCell>
                                    <TableCell className="text-right text-sm font-medium text-primary">
                                      ৳{item.totalTP}
                                    </TableCell>
                                    <TableCell className="text-right text-sm font-medium text-success">
                                      ৳{item.profit}
                                    </TableCell>
                                  </TableRow>
                                ))}
                              </TableBody>
                            </Table>
                          </div>
                        </TableCell>
                      </TableRow>
                    )}
                  </>
                ))
              )}
            </TableBody>
          </Table>
        </div>
      </div>

      <AddDailySaleDialog
        open={addOpen}
        onOpenChange={setAddOpen}
        stockItems={stockItems}
        onSubmit={handleAddSale}
      />

      <InvoiceDialog
        open={invoiceDialogOpen}
        onOpenChange={setInvoiceDialogOpen}
        sale={selectedSale}
      />
    </>
  );
};

export default DailySalesTable;
