import { useState, useEffect } from "react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { StockItem } from "@/data/stockData";
import { Pencil } from "lucide-react";

interface EditStockDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  item: StockItem | null;
  onSave: (item: StockItem) => void;
}

const EditStockDialog = ({ open, onOpenChange, item, onSave }: EditStockDialogProps) => {
  const [form, setForm] = useState({
    itemName: "",
    vendor: "",
    costPrice: "",
    sellingPrice: "",
    stockInDate: "",
  });

  useEffect(() => {
    if (item) {
      setForm({
        itemName: item.itemName,
        vendor: item.vendor || "",
        costPrice: String(item.costPrice),
        sellingPrice: String(item.sellingPrice),
        stockInDate: item.stockInDate,
      });
    }
  }, [item]);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!item || !form.itemName || !form.costPrice || !form.sellingPrice) return;

    const cost = Number(form.costPrice);
    const price = Number(form.sellingPrice);

    onSave({
      ...item,
      itemName: form.itemName,
      vendor: form.vendor || undefined,
      costPrice: cost,
      sellingPrice: price,
      stockInDate: form.stockInDate,
      totalStockRate: item.stockBalance * cost,
      totalSalesRate: item.stockOutQty * price,
    });
    onOpenChange(false);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Pencil className="h-5 w-5 text-primary" />
            Edit Stock Item
          </DialogTitle>
          <DialogDescription>Modify the item details below.</DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="editItemName">Item Name</Label>
            <Input
              id="editItemName"
              value={form.itemName}
              onChange={(e) => setForm({ ...form, itemName: e.target.value })}
              required
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="editVendor">Vendor <span className="text-muted-foreground text-xs">(optional)</span></Label>
            <Input
              id="editVendor"
              value={form.vendor}
              onChange={(e) => setForm({ ...form, vendor: e.target.value })}
              placeholder="e.g. ABC Suppliers"
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="editStockInDate">Stock In Date</Label>
            <Input
              id="editStockInDate"
              value={form.stockInDate}
              onChange={(e) => setForm({ ...form, stockInDate: e.target.value })}
            />
          </div>
          <div className="grid grid-cols-2 gap-3">
            <div className="space-y-2">
              <Label htmlFor="editCostPrice">Cost Price (৳)</Label>
              <Input
                id="editCostPrice"
                type="number"
                min="0"
                step="0.01"
                value={form.costPrice}
                onChange={(e) => setForm({ ...form, costPrice: e.target.value })}
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="editSellingPrice">Selling Price (৳)</Label>
              <Input
                id="editSellingPrice"
                type="number"
                min="0"
                step="0.01"
                value={form.sellingPrice}
                onChange={(e) => setForm({ ...form, sellingPrice: e.target.value })}
                required
              />
            </div>
          </div>
          <DialogFooter>
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit">Save Changes</Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  );
};

export default EditStockDialog;
