import { useState, useEffect } from "react";
import {
  LayoutDashboard,
  Package,
  ShoppingCart,
  BarChart3,
  FileText,
  Settings,
  Leaf,
  PanelLeftClose,
  Tag,
  ShoppingBag,
  PanelLeftOpen,
  X,
  LogOut,
} from "lucide-react";
import { useIsMobile } from "@/hooks/use-mobile";
import { useAuth } from "@/contexts/AuthContext";

interface SidebarProps {
  activeTab: string;
  onTabChange: (tab: string) => void;
  collapsed: boolean;
  onToggle: () => void;
  mobileOpen: boolean;
  onMobileClose: () => void;
}

const navItems = [
  { id: "overview", label: "Overview", icon: LayoutDashboard },
  { id: "inventory", label: "Inventory", icon: Package },
  { id: "products", label: "Products", icon: ShoppingBag },
  { id: "categories", label: "Categories", icon: Tag },
  { id: "sales", label: "Daily Sales", icon: ShoppingCart },
  { id: "analytics", label: "Analytics", icon: BarChart3 },
  { id: "invoice", label: "Invoice", icon: FileText },
];

const Sidebar = ({ activeTab, onTabChange, collapsed, onToggle, mobileOpen, onMobileClose }: SidebarProps) => {
  const isMobile = useIsMobile();
  const { logout } = useAuth();

  const handleNavClick = (id: string) => {
    onTabChange(id);
    if (isMobile) onMobileClose();
  };

  const handleLogout = async () => {
    await logout();
  };

  // Prevent body scroll when mobile sidebar is open
  useEffect(() => {
    if (mobileOpen && isMobile) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "";
    }
    return () => { document.body.style.overflow = ""; };
  }, [mobileOpen, isMobile]);

  const sidebarContent = (
    <>
      {/* Logo */}
      <div className={`flex items-center border-b border-sidebar-border ${collapsed && !isMobile ? "justify-center px-2 py-5" : "gap-3 px-6 py-5"}`}>
        <div className="bg-sidebar-primary rounded-lg p-2 flex-shrink-0">
          <Leaf className="h-5 w-5 text-sidebar-primary-foreground" />
        </div>
        {(!collapsed || isMobile) && (
          <div className="min-w-0">
            <h1 className="text-base font-bold font-display text-sidebar-foreground truncate">StockFlow</h1>
            <p className="text-[11px] text-sidebar-foreground/50 truncate">Product Management</p>
          </div>
        )}
        {/* Mobile close button */}
        {isMobile && (
          <button onClick={onMobileClose} className="ml-auto p-1.5 rounded-lg text-sidebar-foreground/70 hover:text-sidebar-foreground hover:bg-sidebar-accent transition-colors">
            <X className="h-5 w-5" />
          </button>
        )}
      </div>

      {/* Toggle button (desktop only) */}
      {!isMobile && (
        <div className={`px-3 pt-3 ${collapsed ? "flex justify-center" : "flex justify-end"}`}>
          <button
            onClick={onToggle}
            className="p-1.5 rounded-lg text-sidebar-foreground/50 hover:text-sidebar-foreground hover:bg-sidebar-accent transition-all duration-200"
            title={collapsed ? "Expand sidebar" : "Collapse sidebar"}
          >
            {collapsed ? <PanelLeftOpen className="h-4 w-4" /> : <PanelLeftClose className="h-4 w-4" />}
          </button>
        </div>
      )}

      {/* Navigation */}
      <nav className="flex-1 px-3 py-3 space-y-1">
        {navItems.map((item) => {
          const isActive = activeTab === item.id;
          return (
            <button
              key={item.id}
              onClick={() => handleNavClick(item.id)}
              title={collapsed && !isMobile ? item.label : undefined}
              className={`w-full flex items-center rounded-lg text-sm font-medium transition-all duration-200 ${
                collapsed && !isMobile ? "justify-center px-2 py-2.5" : "gap-3 px-4 py-2.5"
              } ${
                isActive
                  ? "bg-sidebar-primary text-sidebar-primary-foreground shadow-md"
                  : "text-sidebar-foreground/70 hover:text-sidebar-foreground hover:bg-sidebar-accent"
              }`}
            >
              <item.icon className="h-4 w-4 flex-shrink-0" />
              {(!collapsed || isMobile) && <span className="truncate">{item.label}</span>}
            </button>
          );
        })}
      </nav>

      {/* Footer */}
      <div className="px-3 py-4 border-t border-sidebar-border space-y-1">
        <button
          onClick={handleLogout}
          title={collapsed && !isMobile ? "Logout" : undefined}
          className={`w-full flex items-center rounded-lg text-sm text-sidebar-foreground/70 hover:text-sidebar-foreground hover:bg-destructive/10 hover:text-destructive transition-all duration-200 ${
            collapsed && !isMobile ? "justify-center px-2 py-2.5" : "gap-3 px-4 py-2.5"
          }`}
        >
          <LogOut className="h-4 w-4 flex-shrink-0" />
          {(!collapsed || isMobile) && <span>Logout</span>}
        </button>
        <button
          title={collapsed && !isMobile ? "Settings" : undefined}
          className={`w-full flex items-center rounded-lg text-sm text-sidebar-foreground/70 hover:text-sidebar-foreground hover:bg-sidebar-accent transition-all duration-200 ${
            collapsed && !isMobile ? "justify-center px-2 py-2.5" : "gap-3 px-4 py-2.5"
          }`}
        >
          <Settings className="h-4 w-4 flex-shrink-0" />
          {(!collapsed || isMobile) && <span>Settings</span>}
        </button>
        {(!collapsed || isMobile) && (
          <div className="mt-3 px-4">
            <p className="text-[10px] text-sidebar-foreground/40">Nagar Gram Service</p>
            <p className="text-[10px] text-sidebar-foreground/30">v1.0.0</p>
          </div>
        )}
      </div>
    </>
  );

  // Mobile: slide-over drawer
  if (isMobile) {
    return (
      <>
        {/* Overlay */}
        {mobileOpen && (
          <div
            className="fixed inset-0 bg-foreground/40 backdrop-blur-sm z-50 transition-opacity"
            onClick={onMobileClose}
          />
        )}
        {/* Drawer */}
        <aside
          className={`sidebar-gradient fixed left-0 top-0 h-screen w-72 flex flex-col z-[60] border-r border-sidebar-border transition-transform duration-300 ease-in-out ${
            mobileOpen ? "translate-x-0" : "-translate-x-full"
          }`}
        >
          {sidebarContent}
        </aside>
      </>
    );
  }

  // Desktop: collapsible fixed sidebar
  return (
    <aside
      className={`sidebar-gradient h-screen fixed left-0 top-0 flex flex-col z-50 border-r border-sidebar-border transition-all duration-300 ease-in-out ${
        collapsed ? "w-[60px]" : "w-64"
      }`}
    >
      {sidebarContent}
    </aside>
  );
};

export default Sidebar;
