// Product Stock Management Data extracted from spreadsheet

export interface Product {
  id: number;
  name: string;
  nickname: string;
  category: string;
}

export interface StockItem {
  itemName: string;
  vendor?: string;
  stockInDate: string;
  stockInQty: number;
  stockOutDate: string;
  stockOutQty: number;
  costPrice: number;
  stockBalance: number;
  sellingPrice: number;
  totalSalesRate: number;
  totalStockRate: number;
}

export interface DailySale {
  date: string;
  items: DailySaleItem[];
  totalTP: number;
  totalDP: number;
  totalProfit: number;
  dueAmount: number;
  paidAmount: number;
  discount: number;
  netProfit: number;
}

export interface DailySaleItem {
  itemName: string;
  qty: number;
  dp: number;
  tp: number;
  totalTP: number;
  totalDP: number;
  profit: number;
}

export const products: Product[] = [
  { id: 1001, name: "Pen (3/-)", nickname: "Pen", category: "Stationery" },
  { id: 1002, name: "Design Roller", nickname: "Dn Roller", category: "Stationery" },
  { id: 1003, name: "Design Scale", nickname: "Dn Scale", category: "Stationery" },
  { id: 1004, name: "Large Scale", nickname: "Large Scale", category: "Stationery" },
  { id: 1005, name: "Uno Card", nickname: "Uno Card", category: "Games" },
  { id: 1006, name: "Pokimon Card", nickname: "Pok Card", category: "Games" },
  { id: 1007, name: "BTS Card", nickname: "BTS Card", category: "Games" },
  { id: 1008, name: "Baby Tab (Writing Practice)", nickname: "Baby Tab", category: "Kids" },
  { id: 1009, name: "3D Cute Cartoon Protective Case", nickname: "3D Case", category: "Accessories" },
  { id: 1010, name: "High Quality Kitchen Toy (Kids)", nickname: "Kitchen Toy", category: "Kids" },
  { id: 1011, name: "Low Category Small Bag", nickname: "Small Bag", category: "Bags" },
  { id: 1012, name: "Cartoon Style Kids Bag", nickname: "Kids Bag", category: "Bags" },
  { id: 1013, name: "Cute Charger Cover", nickname: "Charger Cover", category: "Accessories" },
  { id: 1014, name: "X 0", nickname: "X 0", category: "Games" },
  { id: 1015, name: "Tennis Ball", nickname: "Tennis Ball", category: "Sports" },
  { id: 1016, name: "Seasonal Game", nickname: "Seasonal Game", category: "Games" },
  { id: 1017, name: "Badminton", nickname: "Batt", category: "Sports" },
  { id: 1018, name: "Football", nickname: "Football", category: "Sports" },
  { id: 1022, name: "Airpod", nickname: "Airpod", category: "Electronics" },
  { id: 1023, name: "Nick Band", nickname: "Nick Band", category: "Electronics" },
  { id: 1024, name: "Gaming Head Phone", nickname: "Gaming Earphone", category: "Electronics" },
  { id: 1025, name: "Bluetooth Speaker", nickname: "BT Speaker", category: "Electronics" },
  { id: 1026, name: "Mobile Stand", nickname: "Mobile Stand", category: "Accessories" },
  { id: 1027, name: "Power Bank", nickname: "Power Bank", category: "Electronics" },
  { id: 1028, name: "Digital Hand Weigher", nickname: "Hand Weigher", category: "Electronics" },
  { id: 1029, name: "Smart Watch", nickname: "Smart Watch", category: "Electronics" },
];

export const stockItems: StockItem[] = [
  { itemName: "Pen InVisible", stockInDate: "27/12/2025", stockInQty: 111, stockOutDate: "15/1/2026", stockOutQty: 58, costPrice: 15, stockBalance: 53, sellingPrice: 20, totalSalesRate: 1160, totalStockRate: 795 },
  { itemName: "Design Roller", stockInDate: "27/12/2025", stockInQty: 80, stockOutDate: "15/1/2026", stockOutQty: 37, costPrice: 15, stockBalance: 43, sellingPrice: 25, totalSalesRate: 925, totalStockRate: 645 },
  { itemName: "Naruto UNO Card L", stockInDate: "27/12/2025", stockInQty: 10, stockOutDate: "15/1/2026", stockOutQty: 8, costPrice: 38, stockBalance: 2, sellingPrice: 55, totalSalesRate: 440, totalStockRate: 76 },
  { itemName: "Uno Card (Large)", stockInDate: "27/12/2025", stockInQty: 20, stockOutDate: "15/1/2026", stockOutQty: 2, costPrice: 35, stockBalance: 18, sellingPrice: 45, totalSalesRate: 90, totalStockRate: 630 },
  { itemName: "Uno Card (Small)", stockInDate: "27/12/2025", stockInQty: 20, stockOutDate: "15/1/2026", stockOutQty: 6, costPrice: 26, stockBalance: 14, sellingPrice: 35, totalSalesRate: 210, totalStockRate: 364 },
  { itemName: "Pokimon Card (Large)", stockInDate: "27/12/2025", stockInQty: 100, stockOutDate: "15/1/2026", stockOutQty: 100, costPrice: 7.5, stockBalance: 0, sellingPrice: 12, totalSalesRate: 1200, totalStockRate: 0 },
  { itemName: "Pokimon Card (Small)", stockInDate: "27/12/2025", stockInQty: 200, stockOutDate: "15/1/2026", stockOutQty: 58, costPrice: 3.5, stockBalance: 142, sellingPrice: 5, totalSalesRate: 290, totalStockRate: 497 },
  { itemName: "Naruto Card L", stockInDate: "27/12/2025", stockInQty: 130, stockOutDate: "15/1/2026", stockOutQty: 82, costPrice: 7.5, stockBalance: 48, sellingPrice: 12, totalSalesRate: 984, totalStockRate: 360 },
  { itemName: "Naruto Card S", stockInDate: "27/12/2025", stockInQty: 100, stockOutDate: "15/1/2026", stockOutQty: 91, costPrice: 4, stockBalance: 9, sellingPrice: 7, totalSalesRate: 637, totalStockRate: 36 },
  { itemName: "Journal Note L", stockInDate: "27/12/2025", stockInQty: 40, stockOutDate: "15/1/2026", stockOutQty: 34, costPrice: 48, stockBalance: 6, sellingPrice: 60, totalSalesRate: 2040, totalStockRate: 288 },
  { itemName: "Journal Note S", stockInDate: "27/12/2025", stockInQty: 20, stockOutDate: "15/1/2026", stockOutQty: 0, costPrice: 22, stockBalance: 20, sellingPrice: 28, totalSalesRate: 0, totalStockRate: 440 },
  { itemName: "BTS Note M", stockInDate: "27/12/2025", stockInQty: 20, stockOutDate: "15/1/2026", stockOutQty: 2, costPrice: 28, stockBalance: 18, sellingPrice: 40, totalSalesRate: 80, totalStockRate: 504 },
  { itemName: "BTS Note S", stockInDate: "27/12/2025", stockInQty: 20, stockOutDate: "15/1/2026", stockOutQty: 0, costPrice: 15, stockBalance: 20, sellingPrice: 25, totalSalesRate: 0, totalStockRate: 300 },
  { itemName: "Naruto Note S", stockInDate: "27/12/2025", stockInQty: 20, stockOutDate: "15/1/2026", stockOutQty: 12, costPrice: 15, stockBalance: 8, sellingPrice: 25, totalSalesRate: 300, totalStockRate: 120 },
  { itemName: "X0", stockInDate: "27/12/2025", stockInQty: 100, stockOutDate: "15/1/2026", stockOutQty: 56, costPrice: 17, stockBalance: 44, sellingPrice: 30, totalSalesRate: 1680, totalStockRate: 748 },
  { itemName: "Ball Game", stockInDate: "27/12/2025", stockInQty: 120, stockOutDate: "15/1/2026", stockOutQty: 31, costPrice: 28, stockBalance: 89, sellingPrice: 40, totalSalesRate: 1240, totalStockRate: 2492 },
  { itemName: "Sticky Note", stockInDate: "27/12/2025", stockInQty: 60, stockOutDate: "15/1/2026", stockOutQty: 60, costPrice: 22, stockBalance: 0, sellingPrice: 25, totalSalesRate: 1500, totalStockRate: 0 },
  { itemName: "Mobile Stand (Bangla)", stockInDate: "27/12/2025", stockInQty: 50, stockOutDate: "15/1/2026", stockOutQty: 18, costPrice: 4, stockBalance: 32, sellingPrice: 8, totalSalesRate: 144, totalStockRate: 128 },
  { itemName: "Mobile Stand (China)", stockInDate: "27/12/2025", stockInQty: 5, stockOutDate: "15/1/2026", stockOutQty: 4, costPrice: 60, stockBalance: 1, sellingPrice: 60, totalSalesRate: 240, totalStockRate: 60 },
  { itemName: "Mobile Stand (Metal)", stockInDate: "27/12/2025", stockInQty: 5, stockOutDate: "15/1/2026", stockOutQty: 0, costPrice: 60, stockBalance: 5, sellingPrice: 0, totalSalesRate: 0, totalStockRate: 300 },
  { itemName: "Finger Sliper (1 Color)", stockInDate: "27/12/2025", stockInQty: 900, stockOutDate: "15/1/2026", stockOutQty: 353, costPrice: 4.4, stockBalance: 547, sellingPrice: 7, totalSalesRate: 2471, totalStockRate: 2406.8 },
  { itemName: "Cable Type B/C", stockInDate: "27/12/2025", stockInQty: 40, stockOutDate: "15/1/2026", stockOutQty: 0, costPrice: 30, stockBalance: 40, sellingPrice: 0, totalSalesRate: 15631, totalStockRate: 1200 },
];

export const dailySales: DailySale[] = [
  {
    date: "15/01/2026",
    totalTP: 1300, totalDP: 944, totalProfit: 356,
    dueAmount: 0, paidAmount: 0, discount: 0, netProfit: 356,
    items: [
      { itemName: "Pokimon Card (Large)", qty: 10, dp: 8, tp: 12, totalTP: 120, totalDP: 80, profit: 40 },
      { itemName: "Pokimon Card (Small)", qty: 10, dp: 4, tp: 7, totalTP: 70, totalDP: 40, profit: 30 },
      { itemName: "Journal Note L", qty: 2, dp: 49, tp: 60, totalTP: 120, totalDP: 98, profit: 22 },
      { itemName: "Ball Game", qty: 1, dp: 29, tp: 40, totalTP: 40, totalDP: 29, profit: 11 },
      { itemName: "Sticky Note", qty: 18, dp: 22, tp: 25, totalTP: 450, totalDP: 396, profit: 54 },
      { itemName: "Mobile Stand (China)", qty: 1, dp: 61, tp: 80, totalTP: 80, totalDP: 61, profit: 19 },
      { itemName: "Finger Sliper (1 Color)", qty: 60, dp: 4, tp: 7, totalTP: 420, totalDP: 240, profit: 180 },
    ],
  },
  {
    date: "13/01/2026",
    totalTP: 2537, totalDP: 1861, totalProfit: 676,
    dueAmount: 0, paidAmount: 2000, discount: 40, netProfit: 636,
    items: [
      { itemName: "Pen InVisible", qty: 6, dp: 15, tp: 20, totalTP: 120, totalDP: 90, profit: 30 },
      { itemName: "Design Roller", qty: 6, dp: 18, tp: 25, totalTP: 150, totalDP: 108, profit: 42 },
      { itemName: "Pokimon Card (Large)", qty: 23, dp: 8, tp: 12, totalTP: 276, totalDP: 184, profit: 92 },
      { itemName: "Pokimon Card (Small)", qty: 13, dp: 4, tp: 7, totalTP: 91, totalDP: 52, profit: 39 },
      { itemName: "Journal Note L", qty: 13, dp: 49, tp: 60, totalTP: 780, totalDP: 637, profit: 143 },
      { itemName: "X0", qty: 6, dp: 18, tp: 30, totalTP: 180, totalDP: 108, profit: 72 },
      { itemName: "Sticky Note", qty: 14, dp: 22, tp: 25, totalTP: 350, totalDP: 308, profit: 42 },
      { itemName: "Mobile Stand (China)", qty: 2, dp: 61, tp: 80, totalTP: 160, totalDP: 122, profit: 38 },
      { itemName: "Finger Sliper (1 Color)", qty: 50, dp: 4, tp: 7, totalTP: 350, totalDP: 200, profit: 150 },
      { itemName: "Finger Sliper (Multi Color)", qty: 50, dp: 5, tp: 7, totalTP: 350, totalDP: 250, profit: 100 },
    ],
  },
  {
    date: "12/01/2026",
    totalTP: 1689, totalDP: 1211, totalProfit: 478,
    dueAmount: 0, paidAmount: 1445, discount: 15, netProfit: 463,
    items: [
      { itemName: "Design Roller", qty: 2, dp: 18, tp: 25, totalTP: 50, totalDP: 36, profit: 14 },
      { itemName: "Pokimon Card (Large)", qty: 26, dp: 8, tp: 12, totalTP: 312, totalDP: 208, profit: 104 },
      { itemName: "Pokimon Card (Small)", qty: 36, dp: 4, tp: 7, totalTP: 252, totalDP: 144, profit: 108 },
      { itemName: "Journal Note L", qty: 9, dp: 49, tp: 60, totalTP: 540, totalDP: 441, profit: 99 },
      { itemName: "Sticky Note", qty: 2, dp: 22, tp: 25, totalTP: 50, totalDP: 44, profit: 6 },
      { itemName: "Mobile Stand (China)", qty: 2, dp: 61, tp: 80, totalTP: 160, totalDP: 122, profit: 38 },
      { itemName: "Finger Sliper (1 Color)", qty: 30, dp: 4, tp: 7, totalTP: 210, totalDP: 120, profit: 90 },
      { itemName: "BTS L", qty: 3, dp: 44, tp: 55, totalTP: 165, totalDP: 132, profit: 33 },
    ],
  },
  {
    date: "10/01/2026",
    totalTP: 798, totalDP: 532, totalProfit: 266,
    dueAmount: 0, paidAmount: 700, discount: 15, netProfit: 251,
    items: [
      { itemName: "Naruto UNO Card L", qty: 1, dp: 39, tp: 55, totalTP: 55, totalDP: 39, profit: 16 },
      { itemName: "Pokimon Card (Large)", qty: 12, dp: 8, tp: 12, totalTP: 144, totalDP: 96, profit: 48 },
      { itemName: "Pokimon Card (Small)", qty: 12, dp: 4, tp: 7, totalTP: 84, totalDP: 48, profit: 36 },
      { itemName: "Naruto Card S", qty: 10, dp: 4, tp: 7, totalTP: 70, totalDP: 40, profit: 30 },
      { itemName: "X0", qty: 1, dp: 18, tp: 30, totalTP: 30, totalDP: 18, profit: 12 },
      { itemName: "Ball Game", qty: 10, dp: 29, tp: 40, totalTP: 400, totalDP: 290, profit: 110 },
      { itemName: "Finger Sliper (1 Color)", qty: 10, dp: 4, tp: 7, totalTP: 70, totalDP: 40, profit: 30 },
    ],
  },
  {
    date: "08/01/2026",
    totalTP: 710, totalDP: 514, totalProfit: 196,
    dueAmount: 160, paidAmount: 500, discount: 0, netProfit: 196,
    items: [
      { itemName: "Pen InVisible", qty: 5, dp: 15, tp: 20, totalTP: 100, totalDP: 75, profit: 25 },
      { itemName: "Pokimon Card (Large)", qty: 5, dp: 8, tp: 12, totalTP: 60, totalDP: 40, profit: 20 },
      { itemName: "Pokimon Card (Small)", qty: 5, dp: 4, tp: 7, totalTP: 35, totalDP: 20, profit: 15 },
      { itemName: "Naruto Card L", qty: 5, dp: 8, tp: 12, totalTP: 60, totalDP: 40, profit: 20 },
      { itemName: "Naruto Card S", qty: 5, dp: 4, tp: 7, totalTP: 35, totalDP: 20, profit: 15 },
      { itemName: "X0", qty: 5, dp: 18, tp: 30, totalTP: 150, totalDP: 90, profit: 60 },
      { itemName: "Sticky Note", qty: 12, dp: 22, tp: 25, totalTP: 300, totalDP: 264, profit: 36 },
      { itemName: "Finger Sliper (1 Color)", qty: 10, dp: 4, tp: 7, totalTP: 70, totalDP: 40, profit: 30 },
    ],
  },
  {
    date: "07/01/2026",
    totalTP: 2540, totalDP: 1721, totalProfit: 819,
    dueAmount: 0, paidAmount: 2100, discount: 40, netProfit: 779,
    items: [
      { itemName: "Pen InVisible", qty: 18, dp: 15, tp: 20, totalTP: 360, totalDP: 270, profit: 90 },
      { itemName: "Naruto UNO Card L", qty: 8, dp: 39, tp: 55, totalTP: 440, totalDP: 312, profit: 128 },
      { itemName: "Journal Note L", qty: 9, dp: 49, tp: 60, totalTP: 540, totalDP: 441, profit: 99 },
      { itemName: "Ball Game", qty: 2, dp: 29, tp: 40, totalTP: 80, totalDP: 58, profit: 22 },
      { itemName: "Finger Sliper (1 Color)", qty: 160, dp: 4, tp: 7, totalTP: 1120, totalDP: 640, profit: 480 },
    ],
  },
  {
    date: "06/01/2026",
    totalTP: 98, totalDP: 69, totalProfit: 29,
    dueAmount: 0, paidAmount: 0, discount: 0, netProfit: 29,
    items: [
      { itemName: "Pokimon L", qty: 1, dp: 8, tp: 12, totalTP: 12, totalDP: 8, profit: 4 },
      { itemName: "Pokimon S", qty: 1, dp: 4, tp: 7, totalTP: 7, totalDP: 4, profit: 3 },
      { itemName: "Naruto L", qty: 1, dp: 8, tp: 12, totalTP: 12, totalDP: 8, profit: 4 },
      { itemName: "Naruto S", qty: 1, dp: 4, tp: 7, totalTP: 7, totalDP: 4, profit: 3 },
      { itemName: "Invisible Pen", qty: 3, dp: 15, tp: 20, totalTP: 60, totalDP: 45, profit: 15 },
    ],
  },
  {
    date: "05/01/2026",
    totalTP: 1540, totalDP: 1032, totalProfit: 498,
    dueAmount: 654, paidAmount: 0, discount: 0, netProfit: 498,
    items: [
      { itemName: "Pen InVisible", qty: 14, dp: 15, tp: 20, totalTP: 280, totalDP: 210, profit: 70 },
      { itemName: "Design Roller", qty: 8, dp: 18, tp: 25, totalTP: 200, totalDP: 144, profit: 56 },
      { itemName: "Uno Card (Small)", qty: 3, dp: 27, tp: 35, totalTP: 105, totalDP: 81, profit: 24 },
      { itemName: "Pokimon Card (Large)", qty: 3, dp: 8, tp: 12, totalTP: 36, totalDP: 24, profit: 12 },
      { itemName: "Pokimon Card (Small)", qty: 1, dp: 4, tp: 7, totalTP: 7, totalDP: 4, profit: 3 },
      { itemName: "Naruto Card L", qty: 1, dp: 8, tp: 12, totalTP: 12, totalDP: 8, profit: 4 },
      { itemName: "Naruto Card S", qty: 3, dp: 4, tp: 7, totalTP: 21, totalDP: 12, profit: 9 },
      { itemName: "BTS Note S", qty: 5, dp: 16, tp: 25, totalTP: 125, totalDP: 80, profit: 45 },
      { itemName: "X0", qty: 12, dp: 18, tp: 30, totalTP: 360, totalDP: 216, profit: 144 },
      { itemName: "Ball Game", qty: 1, dp: 29, tp: 40, totalTP: 40, totalDP: 29, profit: 11 },
      { itemName: "Sticky Note", qty: 2, dp: 22, tp: 25, totalTP: 50, totalDP: 44, profit: 6 },
      { itemName: "Mobile Stand (Bangla)", qty: 6, dp: 5, tp: 8, totalTP: 48, totalDP: 30, profit: 18 },
      { itemName: "Finger Sliper (Multi Color)", qty: 32, dp: 5, tp: 8, totalTP: 256, totalDP: 160, profit: 96 },
    ],
  },
];

// Computed summary data
export const getTotalProducts = () => products.length;
export const getTotalStockItems = () => stockItems.length;
export const getTotalStockBalance = () => stockItems.reduce((sum, item) => sum + item.stockBalance, 0);
export const getTotalStockValue = () => stockItems.reduce((sum, item) => sum + item.totalStockRate, 0);
export const getTotalSalesValue = () => stockItems.reduce((sum, item) => sum + item.totalSalesRate, 0);
export const getTotalProfit = () => dailySales.reduce((sum, day) => sum + day.totalProfit, 0);
export const getOutOfStockItems = () => stockItems.filter(item => item.stockBalance === 0);
export const getLowStockItems = () => stockItems.filter(item => item.stockBalance > 0 && item.stockBalance <= 10);

export const getCategoryDistribution = () => {
  const categories: Record<string, number> = {};
  products.forEach(p => {
    categories[p.category] = (categories[p.category] || 0) + 1;
  });
  return Object.entries(categories).map(([name, value]) => ({ name, value }));
};

export const getSalesTrend = () => {
  return [...dailySales].reverse().map(day => ({
    date: day.date.replace("/2026", ""),
    sales: day.totalTP,
    cost: day.totalDP,
    profit: day.totalProfit,
  }));
};

export const getTopSellingItems = () => {
  const itemSales: Record<string, number> = {};
  dailySales.forEach(day => {
    day.items.forEach(item => {
      itemSales[item.itemName] = (itemSales[item.itemName] || 0) + item.qty;
    });
  });
  return Object.entries(itemSales)
    .map(([name, qty]) => ({ name, qty }))
    .sort((a, b) => b.qty - a.qty)
    .slice(0, 8);
};
