import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { salesApi, DailySale } from '@/lib/api';
import { toast } from 'sonner';
import { useMemo } from 'react';

export const useDailySales = () => {
  const queryClient = useQueryClient();

  const { data, isLoading, error } = useQuery({
    queryKey: ['daily-sales'],
    queryFn: async () => {
      const response = await salesApi.getAll();
      return response.data.data;
    },
  });

  const addSale = useMutation({
    mutationFn: (sale: DailySale) => salesApi.create(sale),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['daily-sales'] });
      // Also invalidate stock items since auto stock-out happens
      queryClient.invalidateQueries({ queryKey: ['stock-items'] });
      toast.success('Daily sale added successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to add daily sale');
    },
  });

  const deleteSale = useMutation({
    mutationFn: (id: number) => salesApi.delete(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['daily-sales'] });
      toast.success('Daily sale deleted successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to delete daily sale');
    },
  });

  // Calculate totals from data
  const totals = useMemo(() => {
    if (!data) {
      return {
        totalSales: 0,
        totalProfit: 0,
        totalDue: 0,
      };
    }

    return {
      totalSales: data.reduce((sum: number, sale: DailySale) => sum + sale.totalTP, 0),
      totalProfit: data.reduce((sum: number, sale: DailySale) => sum + sale.totalProfit, 0),
      totalDue: data.reduce((sum: number, sale: DailySale) => sum + sale.dueAmount, 0),
    };
  }, [data]);

  return {
    sales: data || [],
    totals,
    isLoading,
    error,
    actions: {
      addSale: (sale: DailySale) => addSale.mutate(sale),
      deleteSale: (id: number) => deleteSale.mutate(id),
    },
  };
};
