import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { productsApi, Product } from '@/lib/api';
import { toast } from 'sonner';

export const useProducts = () => {
  const queryClient = useQueryClient();

  const { data, isLoading, error } = useQuery({
    queryKey: ['products'],
    queryFn: async () => {
      const response = await productsApi.getAll();
      return response.data.data;
    },
  });

  const addProduct = useMutation({
    mutationFn: (product: { name: string; nickname: string; categoryId: number }) => productsApi.create(product),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['products'] });
      toast.success('Product added successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to add product');
    },
  });

  const updateProduct = useMutation({
    mutationFn: ({ id, data }: { id: number; data: { name: string; nickname: string; categoryId: number } }) =>
      productsApi.update(id, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['products'] });
      toast.success('Product updated successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to update product');
    },
  });

  const deleteProduct = useMutation({
    mutationFn: (id: number) => productsApi.delete(id),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['products'] });
      toast.success('Product deleted successfully');
    },
    onError: (error: any) => {
      toast.error(error.response?.data?.message || 'Failed to delete product');
    },
  });

  return {
    productList: data || [],
    isLoading,
    error,
    addProduct: (product: { name: string; nickname: string; categoryId: number }) => addProduct.mutate(product),
    updateProduct: (id: number, product: { name: string; nickname: string; categoryId: number }) =>
      updateProduct.mutate({ id, data: product }),
    deleteProduct: (id: number) => deleteProduct.mutate(id),
  };
};
