import { useState, useCallback, useMemo } from "react";
import { DailySale, DailySaleItem, dailySales as initialDailySales } from "@/data/stockData";

export interface DailySalesManagerActions {
  addSale: (sale: DailySale) => void;
  deleteSale: (date: string) => void;
}

export const useDailySalesManager = () => {
  const [sales, setSales] = useState<DailySale[]>(() => [...initialDailySales]);

  const addSale = useCallback((sale: DailySale) => {
    setSales((prev) => {
      // If a sale for the same date exists, merge items into it
      const existingIndex = prev.findIndex((s) => s.date === sale.date);
      if (existingIndex !== -1) {
        const existing = prev[existingIndex];
        const merged: DailySale = {
          ...existing,
          items: [...existing.items, ...sale.items],
          totalTP: existing.totalTP + sale.totalTP,
          totalDP: existing.totalDP + sale.totalDP,
          totalProfit: existing.totalProfit + sale.totalProfit,
          paidAmount: existing.paidAmount + sale.paidAmount,
          dueAmount: existing.dueAmount + sale.dueAmount,
          discount: existing.discount + sale.discount,
          netProfit: existing.netProfit + sale.netProfit,
        };
        return prev.map((s, i) => (i === existingIndex ? merged : s));
      }
      return [sale, ...prev];
    });
  }, []);

  const deleteSale = useCallback((date: string) => {
    setSales((prev) => prev.filter((s) => s.date !== date));
  }, []);

  const totals = useMemo(() => {
    const totalSales = sales.reduce((sum, s) => sum + s.totalTP, 0);
    const totalProfit = sales.reduce((sum, s) => sum + s.totalProfit, 0);
    const totalDue = sales.reduce((sum, s) => sum + s.dueAmount, 0);
    return { totalSales, totalProfit, totalDue };
  }, [sales]);

  return {
    sales,
    totals,
    actions: { addSale, deleteSale },
  };
};
