import { useState } from "react";
import {
  Package,
  TrendingUp,
  DollarSign,
  AlertTriangle,
  ShoppingCart,
  Boxes,
  Search,
  Bell,
  Menu,
} from "lucide-react";
import { useIsMobile } from "@/hooks/use-mobile";
import { useStockItems } from "@/hooks/queries/useStockItems";
import { useCategories } from "@/hooks/queries/useCategories";
import { useProducts } from "@/hooks/queries/useProducts";
import { useDailySales } from "@/hooks/queries/useDailySales";
import Sidebar from "@/components/dashboard/Sidebar";
import StatCard from "@/components/dashboard/StatCard";
import StockTable from "@/components/dashboard/StockTable";
import SalesChart from "@/components/dashboard/SalesChart";
import TopProductsChart from "@/components/dashboard/TopProductsChart";
import ProfitChart from "@/components/dashboard/ProfitChart";
import DailySalesTable from "@/components/dashboard/DailySalesTable";
import StockOverviewChart from "@/components/dashboard/StockOverviewChart";
import InvoicePreview from "@/components/dashboard/InvoicePreview";
import CategoriesTable from "@/components/dashboard/CategoriesTable";
import ProductsTable from "@/components/dashboard/ProductsTable";

const Index = () => {
  const [activeTab, setActiveTab] = useState("overview");
  const [sidebarCollapsed, setSidebarCollapsed] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const isMobile = useIsMobile();

  const { items: stockItemsList, stats, actions: apiActions } = useStockItems();
  const { categories, addCategory, updateCategory, deleteCategory } = useCategories();
  const { productList: apiProductList, addProduct: apiAddProduct, updateProduct: apiUpdateProduct, deleteProduct: apiDeleteProduct } = useProducts();
  const { sales, totals: salesTotals, actions: salesActions } = useDailySales();

  // Convert API products to component format (categoryId -> category name)
  const productList = apiProductList;

  // Wrapper functions to convert category names to IDs
  const addProduct = (product: { name: string; nickname: string; category: string }) => {
    const category = categories.find(c => c.name === product.category);
    if (category) {
      apiAddProduct({
        name: product.name,
        nickname: product.nickname,
        categoryId: category.id
      });
    }
  };

  const updateProduct = (id: number, product: { name: string; nickname: string; category: string }) => {
    const category = categories.find(c => c.name === product.category);
    if (category) {
      apiUpdateProduct(id, {
        name: product.name,
        nickname: product.nickname,
        categoryId: category.id
      });
    }
  };

  const deleteProduct = (id: number) => apiDeleteProduct(id);

  // Wrapper to convert index-based actions to id-based API calls
  const actions = {
    addItem: (item: any) => apiActions.addItem(item),
    updateItem: (index: number, item: any) => {
      const stockItem = stockItemsList[index];
      if (stockItem?.id) {
        apiActions.updateItem(stockItem.id, item);
      }
    },
    deleteItem: (index: number) => {
      const stockItem = stockItemsList[index];
      if (stockItem?.id) {
        apiActions.deleteItem(stockItem.id);
      }
    },
    stockIn: (index: number, qty: number, date: string) => {
      const stockItem = stockItemsList[index];
      if (stockItem?.id) {
        apiActions.stockIn(stockItem.id, qty, date);
      }
    },
    stockOut: (index: number, qty: number, date: string) => {
      const stockItem = stockItemsList[index];
      if (stockItem?.id) {
        apiActions.stockOut(stockItem.id, qty, date);
      }
    },
  };

  const totalProducts = stats.totalProducts;
  const totalBalance = stats.totalBalance;
  const stockValue = stats.stockValue;
  const salesValue = stats.salesValue;
  const totalProfit = salesTotals.totalProfit;
  const outOfStock = stats.outOfStock;
  const lowStock = stats.lowStock;

  // Handle adding a sale - backend automatically handles stock-out
  const handleAddSale = (
    sale: import("@/data/stockData").DailySale,
    stockOutItems: { itemName: string; qty: number }[]
  ) => {
    // Backend automatically handles stock-out, so just add the sale
    salesActions.addSale(sale);
  };

  // Handle deleting a sale - convert date to id
  const handleDeleteSale = (date: string) => {
    const sale = sales.find((s) => s.date === date);
    if (sale && sale.id) {
      salesActions.deleteSale(sale.id);
    }
  };

  const mainMargin = isMobile ? "ml-0" : sidebarCollapsed ? "ml-[60px]" : "ml-64";

  return (
    <div className="flex min-h-screen bg-background">
      <Sidebar
        activeTab={activeTab}
        onTabChange={setActiveTab}
        collapsed={sidebarCollapsed}
        onToggle={() => setSidebarCollapsed(!sidebarCollapsed)}
        mobileOpen={mobileMenuOpen}
        onMobileClose={() => setMobileMenuOpen(false)}
      />

      <main className={`flex-1 transition-all duration-300 ease-in-out ${mainMargin}`}>
        {/* Top Bar */}
        <header className="sticky top-0 z-40 bg-background/80 backdrop-blur-md border-b border-border px-4 md:px-6 py-3">
          <div className="flex items-center justify-between gap-3">
            <div className="flex items-center gap-3 min-w-0">
              {/* Hamburger menu (mobile) */}
              {isMobile && (
                <button
                  onClick={() => setMobileMenuOpen(true)}
                  className="p-2 rounded-lg bg-muted/50 border border-border hover:bg-muted transition-colors flex-shrink-0"
                >
                  <Menu className="h-5 w-5 text-foreground" />
                </button>
              )}
              <div className="min-w-0">
                <h2 className="text-lg md:text-xl font-bold font-display text-foreground capitalize truncate">
                  {activeTab === "overview" ? "Dashboard" : activeTab}
                </h2>
                <p className="text-xs text-muted-foreground hidden sm:block">
                  Welcome back! Here's your stock summary.
                </p>
              </div>
            </div>
            <div className="flex items-center gap-2 md:gap-3 flex-shrink-0">
              <div className="relative hidden sm:block">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <input
                  type="text"
                  placeholder="Search products..."
                  className="pl-9 pr-4 py-2 text-sm bg-muted/50 border border-border rounded-lg focus:outline-none focus:ring-2 focus:ring-primary/30 focus:border-primary transition-all w-40 lg:w-56"
                />
              </div>
              <button className="relative p-2 rounded-lg bg-muted/50 border border-border hover:bg-muted transition-colors">
                <Bell className="h-4 w-4 text-muted-foreground" />
                {(outOfStock + lowStock) > 0 && (
                  <span className="absolute -top-1 -right-1 h-4 w-4 bg-destructive rounded-full text-[10px] text-destructive-foreground flex items-center justify-center font-bold">
                    {outOfStock + lowStock}
                  </span>
                )}
              </button>
            </div>
          </div>
        </header>

        <div className="p-4 md:p-6">
          {/* Overview Tab */}
          {activeTab === "overview" && (
            <div className="space-y-4 md:space-y-6">
              <div className="grid grid-cols-2 lg:grid-cols-4 gap-3 md:gap-4">
                <StatCard title="Total Products" value={totalProducts} subtitle="Across all categories" icon={Package} delay={0} />
                <StatCard title="Stock Balance" value={totalBalance.toLocaleString()} subtitle="Units in warehouse" icon={Boxes} trend="neutral" trendValue={`${outOfStock} out of stock`} delay={100} />
                <StatCard title="Total Sales" value={`৳${salesValue.toLocaleString()}`} subtitle="Revenue generated" icon={ShoppingCart} trend="up" trendValue="Active sales" delay={200} />
                <StatCard title="Total Profit" value={`৳${totalProfit.toLocaleString()}`} subtitle="Net earnings" icon={TrendingUp} trend="up" trendValue="Growing" delay={300} />
              </div>

              {(outOfStock > 0 || lowStock > 0) && (
                <div className="flex flex-wrap gap-3">
                  {outOfStock > 0 && (
                    <div className="flex items-center gap-2 bg-destructive/10 border border-destructive/20 text-destructive rounded-lg px-3 md:px-4 py-2 text-xs md:text-sm animate-fade-in">
                      <AlertTriangle className="h-4 w-4 flex-shrink-0" />
                      <span className="font-medium">{outOfStock} items out of stock</span>
                    </div>
                  )}
                  {lowStock > 0 && (
                    <div className="flex items-center gap-2 bg-warning/10 border border-warning/20 text-warning rounded-lg px-3 md:px-4 py-2 text-xs md:text-sm animate-fade-in">
                      <AlertTriangle className="h-4 w-4 flex-shrink-0" />
                      <span className="font-medium">{lowStock} items low stock</span>
                    </div>
                  )}
                </div>
              )}

              <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 md:gap-6">
                <SalesChart />
                <TopProductsChart />
              </div>

              <div className="grid grid-cols-1 md:grid-cols-3 gap-3 md:gap-4">
                <StatCard title="Stock Value" value={`৳${stockValue.toLocaleString()}`} subtitle="Current inventory worth" icon={DollarSign} delay={0} />
                <StatCard title="Out of Stock" value={outOfStock} subtitle="Items need restocking" icon={AlertTriangle} trend={outOfStock > 0 ? "down" : "neutral"} trendValue={outOfStock > 0 ? "Needs attention" : "All stocked"} delay={100} />
                <StatCard title="Low Stock" value={lowStock} subtitle="Below 10 units" icon={AlertTriangle} trend={lowStock > 0 ? "down" : "neutral"} trendValue={lowStock > 0 ? "Order soon" : "All good"} delay={200} />
              </div>
            </div>
          )}

          {activeTab === "inventory" && (
            <div className="space-y-6">
              <StockTable items={stockItemsList} actions={actions} />
            </div>
          )}

          {activeTab === "products" && (
            <div className="space-y-6">
              <ProductsTable
                products={productList}
                categories={categories}
                onAdd={addProduct}
                onUpdate={updateProduct}
                onDelete={deleteProduct}
              />
            </div>
          )}

          {activeTab === "categories" && (
            <div className="space-y-6">
              <CategoriesTable
                categories={categories}
                onAdd={addCategory}
                onUpdate={updateCategory}
                onDelete={deleteCategory}
              />
            </div>
          )}

          {activeTab === "sales" && (
            <div className="space-y-6">
              <DailySalesTable
                sales={sales}
                stockItems={stockItemsList}
                onAddSale={handleAddSale}
                onDeleteSale={handleDeleteSale}
              />
            </div>
          )}

          {activeTab === "analytics" && (
            <div className="space-y-4 md:space-y-6">
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 md:gap-6">
                <SalesChart />
                <ProfitChart />
              </div>
              <StockOverviewChart />
              <TopProductsChart />
            </div>
          )}

          {activeTab === "invoice" && (
            <div className="space-y-6">
              <InvoicePreview />
            </div>
          )}
        </div>
      </main>
    </div>
  );
};

export default Index;
